'use server';

/**
 * @fileOverview A Genkit flow that generates a mobile-optimized layout for a digital menu.
 *
 * - generateMobileOptimizedLayout - A function that generates a mobile-optimized layout for a digital menu.
 * - GenerateMobileOptimizedLayoutInput - The input type for the generateMobileOptimizedLayout function.
 * - GenerateMobileOptimizedLayoutOutput - The return type for the generateMobileOptimizedLayout function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const GenerateMobileOptimizedLayoutInputSchema = z.object({
  menuData: z.string().describe('The JSON data representing the digital menu.'),
  brandColors: z
    .string()
    .describe(
      'A description of the brand colors to use, including primary, background, and accent colors.'
    ),
  typography: z.string().describe('The desired typography for the menu.'),
});
export type GenerateMobileOptimizedLayoutInput = z.infer<
  typeof GenerateMobileOptimizedLayoutInputSchema
>;

const GenerateMobileOptimizedLayoutOutputSchema = z.object({
  mobileOptimizedLayout: z
    .string()
    .describe(
      'A JSON string representing the mobile-optimized layout for the digital menu.
      It should include specifications for element alignment, font sizes, image scaling, and any other visual adjustments
      necessary to ensure it looks professional on all mobile devices.'
    ),
});
export type GenerateMobileOptimizedLayoutOutput = z.infer<
  typeof GenerateMobileOptimizedLayoutOutputSchema
>;

export async function generateMobileOptimizedLayout(
  input: GenerateMobileOptimizedLayoutInput
): Promise<GenerateMobileOptimizedLayoutOutput> {
  return generateMobileOptimizedLayoutFlow(input);
}

const prompt = ai.definePrompt({
  name: 'generateMobileOptimizedLayoutPrompt',
  input: {schema: GenerateMobileOptimizedLayoutInputSchema},
  output: {schema: GenerateMobileOptimizedLayoutOutputSchema},
  prompt: `You are an expert in mobile UI/UX design, specializing in optimizing digital menus for restaurants.

  Given the following digital menu data, brand colors, and typography preferences, generate a mobile-optimized layout.

  Menu Data:
  {{menuData}}

  Brand Colors:
  {{brandColors}}

  Typography:
  {{typography}}

  Consider:
  - A mobile-first design with a single-column layout.
  - Neat alignment of all elements for a professional look.
  - Usage of recognizable and vibrant icons for categories and product options.

  Ensure the layout is responsive and adapts well to various mobile screen sizes. Return the layout as a JSON string.
  The JSON string should include specifications for:
  - Element alignment
  - Font sizes
  - Image scaling
  - Any other visual adjustments necessary to ensure it looks professional on all mobile devices.

  Output the mobile-optimized layout as a JSON string.
  `,
});

const generateMobileOptimizedLayoutFlow = ai.defineFlow(
  {
    name: 'generateMobileOptimizedLayoutFlow',
    inputSchema: GenerateMobileOptimizedLayoutInputSchema,
    outputSchema: GenerateMobileOptimizedLayoutOutputSchema,
  },
  async input => {
    const {output} = await prompt(input);
    return output!;
  }
);
